/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.reflect;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.base.StandardSystemProperty;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import com.google.common.io.ByteSource;
import com.google.common.io.CharSource;
import com.google.common.io.Resources;
import com.google.common.reflect.Reflection;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.checkerframework.checker.nullness.qual.Nullable;

@Beta
public final class ClassPath {
    private static final Logger logger = Logger.getLogger(ClassPath.class.getName());
    private static final Predicate<ClassInfo> IS_TOP_LEVEL = new Predicate<ClassInfo>(){

        @Override
        public boolean apply(ClassInfo info) {
            return info.className.indexOf(36) == -1;
        }
    };
    private static final Splitter CLASS_PATH_ATTRIBUTE_SEPARATOR = Splitter.on(" ").omitEmptyStrings();
    private static final String CLASS_FILE_NAME_EXTENSION = ".class";
    private final ImmutableSet<ResourceInfo> resources;

    private ClassPath(ImmutableSet<ResourceInfo> resources2) {
        this.resources = resources2;
    }

    public static ClassPath from(ClassLoader classloader) throws IOException {
        DefaultScanner scanner = new DefaultScanner();
        scanner.scan(classloader);
        return new ClassPath(scanner.getResources());
    }

    public ImmutableSet<ResourceInfo> getResources() {
        return this.resources;
    }

    public ImmutableSet<ClassInfo> getAllClasses() {
        return FluentIterable.from(this.resources).filter(ClassInfo.class).toSet();
    }

    public ImmutableSet<ClassInfo> getTopLevelClasses() {
        return FluentIterable.from(this.resources).filter(ClassInfo.class).filter(IS_TOP_LEVEL).toSet();
    }

    public ImmutableSet<ClassInfo> getTopLevelClasses(String packageName) {
        Preconditions.checkNotNull(packageName);
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (ClassInfo classInfo : this.getTopLevelClasses()) {
            if (!classInfo.getPackageName().equals(packageName)) continue;
            builder.add(classInfo);
        }
        return builder.build();
    }

    public ImmutableSet<ClassInfo> getTopLevelClassesRecursive(String packageName) {
        Preconditions.checkNotNull(packageName);
        String packagePrefix = new StringBuilder(1 + String.valueOf(packageName).length()).append(packageName).append('.').toString();
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (ClassInfo classInfo : this.getTopLevelClasses()) {
            if (!classInfo.getName().startsWith(packagePrefix)) continue;
            builder.add(classInfo);
        }
        return builder.build();
    }

    @VisibleForTesting
    static String getClassName(String filename) {
        int classNameEnd = filename.length() - CLASS_FILE_NAME_EXTENSION.length();
        return filename.substring(0, classNameEnd).replace('/', '.');
    }

    @VisibleForTesting
    static File toFile(URL url2) {
        Preconditions.checkArgument(url2.getProtocol().equals("file"));
        try {
            return new File(url2.toURI());
        }
        catch (URISyntaxException e) {
            return new File(url2.getPath());
        }
    }

    @VisibleForTesting
    static final class DefaultScanner
    extends Scanner {
        private final SetMultimap<ClassLoader, ResourceInfo> resources = MultimapBuilder.hashKeys().linkedHashSetValues().build();

        DefaultScanner() {
        }

        ImmutableSet<ResourceInfo> getResources() {
            return ImmutableSet.copyOf(this.resources.values());
        }

        @Override
        protected void scanResource(ResourceInfo resource2) {
            this.resources.put(resource2.loader, resource2);
        }
    }

    static abstract class Scanner {
        private final Set<File> scannedUris = Sets.newConcurrentHashSet();

        Scanner() {
        }

        public final void scan(ClassLoader classloader) throws IOException {
            for (Map.Entry entry : Scanner.getClassPathEntries(classloader).entrySet()) {
                this.scan((File)entry.getKey(), (ClassLoader)entry.getValue());
            }
        }

        @VisibleForTesting
        final void scan(File file2, ClassLoader classloader) throws IOException {
            if (this.scannedUris.add(file2.getCanonicalFile())) {
                this.scanFrom(file2, classloader);
            }
        }

        protected abstract void scanResource(ResourceInfo var1) throws IOException;

        protected void scanFrom(File file2, ClassLoader classloader) throws IOException {
            try {
                if (!file2.exists()) {
                    return;
                }
            }
            catch (SecurityException e) {
                String string = String.valueOf(file2);
                String string2 = String.valueOf(e);
                logger.warning(new StringBuilder(16 + String.valueOf(string).length() + String.valueOf(string2).length()).append("Cannot access ").append(string).append(": ").append(string2).toString());
                return;
            }
            if (file2.isDirectory()) {
                this.scanDirectory(classloader, file2);
            } else {
                this.scanJar(file2, classloader);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void scanJar(File file2, ClassLoader classloader) throws IOException {
            JarFile jarFile;
            try {
                jarFile = new JarFile(file2);
            }
            catch (IOException e) {
                return;
            }
            try {
                for (File path : Scanner.getClassPathFromManifest(file2, jarFile.getManifest())) {
                    this.scan(path, classloader);
                }
                this.scanJarFile(classloader, jarFile);
            }
            finally {
                try {
                    jarFile.close();
                }
                catch (IOException iOException) {}
            }
        }

        @VisibleForTesting
        static ImmutableSet<File> getClassPathFromManifest(File jarFile, @Nullable Manifest manifest) {
            if (manifest == null) {
                return ImmutableSet.of();
            }
            ImmutableSet.Builder builder = ImmutableSet.builder();
            String classpathAttribute = manifest.getMainAttributes().getValue(Attributes.Name.CLASS_PATH.toString());
            if (classpathAttribute != null) {
                for (String path : CLASS_PATH_ATTRIBUTE_SEPARATOR.split(classpathAttribute)) {
                    URL url2;
                    try {
                        url2 = Scanner.getClassPathEntry(jarFile, path);
                    }
                    catch (MalformedURLException e) {
                        String string = String.valueOf(path);
                        logger.warning(string.length() != 0 ? "Invalid Class-Path entry: ".concat(string) : new String("Invalid Class-Path entry: "));
                        continue;
                    }
                    if (!url2.getProtocol().equals("file")) continue;
                    builder.add(ClassPath.toFile(url2));
                }
            }
            return builder.build();
        }

        @VisibleForTesting
        static ImmutableMap<File, ClassLoader> getClassPathEntries(ClassLoader classloader) {
            LinkedHashMap<File, ClassLoader> entries2 = Maps.newLinkedHashMap();
            ClassLoader parent = classloader.getParent();
            if (parent != null) {
                entries2.putAll(Scanner.getClassPathEntries(parent));
            }
            for (URL url2 : Scanner.getClassLoaderUrls(classloader)) {
                File file2;
                if (!url2.getProtocol().equals("file") || entries2.containsKey(file2 = ClassPath.toFile(url2))) continue;
                entries2.put(file2, classloader);
            }
            return ImmutableMap.copyOf(entries2);
        }

        private static ImmutableList<URL> getClassLoaderUrls(ClassLoader classloader) {
            if (classloader instanceof URLClassLoader) {
                return ImmutableList.copyOf(((URLClassLoader)classloader).getURLs());
            }
            if (classloader.equals(ClassLoader.getSystemClassLoader())) {
                return Scanner.parseJavaClassPath();
            }
            return ImmutableList.of();
        }

        @VisibleForTesting
        static ImmutableList<URL> parseJavaClassPath() {
            ImmutableList.Builder urls2 = ImmutableList.builder();
            for (String entry : Splitter.on(StandardSystemProperty.PATH_SEPARATOR.value()).split(StandardSystemProperty.JAVA_CLASS_PATH.value())) {
                try {
                    try {
                        urls2.add(new File(entry).toURI().toURL());
                    }
                    catch (SecurityException e) {
                        urls2.add(new URL("file", null, new File(entry).getAbsolutePath()));
                    }
                }
                catch (MalformedURLException e) {
                    String string = String.valueOf(entry);
                    logger.log(Level.WARNING, string.length() != 0 ? "malformed classpath entry: ".concat(string) : new String("malformed classpath entry: "), e);
                }
            }
            return urls2.build();
        }

        @VisibleForTesting
        static URL getClassPathEntry(File jarFile, String path) throws MalformedURLException {
            return new URL(jarFile.toURI().toURL(), path);
        }

        @VisibleForTesting
        void scanJarFile(ClassLoader classloader, JarFile file2) throws IOException {
            Enumeration<JarEntry> entries2 = file2.entries();
            while (entries2.hasMoreElements()) {
                JarEntry entry = entries2.nextElement();
                if (entry.isDirectory() || entry.getName().equals("META-INF/MANIFEST.MF")) continue;
                this.scanResource(ResourceInfo.of(new File(file2.getName()), entry.getName(), classloader));
            }
        }

        private void scanDirectory(ClassLoader classloader, File directory) throws IOException {
            HashSet<File> currentPath = new HashSet<File>();
            currentPath.add(directory.getCanonicalFile());
            this.scanDirectory(directory, classloader, "", currentPath);
        }

        private void scanDirectory(File directory, ClassLoader classloader, String packagePrefix, Set<File> currentPath) throws IOException {
            File[] files2 = directory.listFiles();
            if (files2 == null) {
                String string = String.valueOf(directory);
                logger.warning(new StringBuilder(22 + String.valueOf(string).length()).append("Cannot read directory ").append(string).toString());
                return;
            }
            for (File f : files2) {
                String resourceName;
                String name = f.getName();
                if (f.isDirectory()) {
                    File deref = f.getCanonicalFile();
                    if (!currentPath.add(deref)) continue;
                    this.scanDirectory(deref, classloader, new StringBuilder(1 + String.valueOf(packagePrefix).length() + String.valueOf(name).length()).append(packagePrefix).append(name).append("/").toString(), currentPath);
                    currentPath.remove(deref);
                    continue;
                }
                String string = String.valueOf(packagePrefix);
                String string2 = String.valueOf(name);
                String string3 = resourceName = string2.length() != 0 ? string.concat(string2) : new String(string);
                if (resourceName.equals("META-INF/MANIFEST.MF")) continue;
                this.scanResource(ResourceInfo.of(f, resourceName, classloader));
            }
        }
    }

    @Beta
    public static final class ClassInfo
    extends ResourceInfo {
        private final String className;

        ClassInfo(File file2, String resourceName, ClassLoader loader) {
            super(file2, resourceName, loader);
            this.className = ClassPath.getClassName(resourceName);
        }

        public String getPackageName() {
            return Reflection.getPackageName(this.className);
        }

        public String getSimpleName() {
            int lastDollarSign = this.className.lastIndexOf(36);
            if (lastDollarSign != -1) {
                String innerClassName = this.className.substring(lastDollarSign + 1);
                return CharMatcher.inRange('0', '9').trimLeadingFrom(innerClassName);
            }
            String packageName = this.getPackageName();
            if (packageName.isEmpty()) {
                return this.className;
            }
            return this.className.substring(packageName.length() + 1);
        }

        public String getName() {
            return this.className;
        }

        public Class<?> load() {
            try {
                return this.loader.loadClass(this.className);
            }
            catch (ClassNotFoundException e) {
                throw new IllegalStateException(e);
            }
        }

        @Override
        public String toString() {
            return this.className;
        }
    }

    @Beta
    public static class ResourceInfo {
        private final File file;
        private final String resourceName;
        final ClassLoader loader;

        static ResourceInfo of(File file2, String resourceName, ClassLoader loader) {
            if (resourceName.endsWith(ClassPath.CLASS_FILE_NAME_EXTENSION)) {
                return new ClassInfo(file2, resourceName, loader);
            }
            return new ResourceInfo(file2, resourceName, loader);
        }

        ResourceInfo(File file2, String resourceName, ClassLoader loader) {
            this.file = Preconditions.checkNotNull(file2);
            this.resourceName = Preconditions.checkNotNull(resourceName);
            this.loader = Preconditions.checkNotNull(loader);
        }

        public final URL url() {
            URL url2 = this.loader.getResource(this.resourceName);
            if (url2 == null) {
                throw new NoSuchElementException(this.resourceName);
            }
            return url2;
        }

        public final ByteSource asByteSource() {
            return Resources.asByteSource(this.url());
        }

        public final CharSource asCharSource(Charset charset) {
            return Resources.asCharSource(this.url(), charset);
        }

        public final String getResourceName() {
            return this.resourceName;
        }

        final File getFile() {
            return this.file;
        }

        public int hashCode() {
            return this.resourceName.hashCode();
        }

        public boolean equals(Object obj) {
            if (obj instanceof ResourceInfo) {
                ResourceInfo that = (ResourceInfo)obj;
                return this.resourceName.equals(that.resourceName) && this.loader == that.loader;
            }
            return false;
        }

        public String toString() {
            return this.resourceName;
        }
    }
}

